//*****************************************************************************
// Description     : Main sample program of histogram mode
// Target Device   : APG7400A
// Target Platform : Linux, C
// Date            : June 2025
// Version         : 2.0
//*****************************************************************************
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "ftd2xx.h"
#include "usbmca.h"

//
// Modify the following defines according to your environment
//
/******************************************************************************/
#define     MEASURING_TIME      (3000llu)   /* msec */
#define     SAVE_FILE           "./hist_data.csv"  // name of the file where histogram data saved
/******************************************************************************/
static int  file_save_histogram(char*, USBMCA_HISTOGRAM*) ;

//#########################################################################
//  Name      : main
//  Function  : entry point
//  Arguments :
//      in argc : number of parameters
//      in argv : parameters
//  Return :
//      0 : success
//      non-0 : failure
//  Note : 
//#########################################################################
int main(int argc, char **argv)
{
	USBMCA_HISTOGRAM    histdt;
	FT_STATUS   ret0th, ret1st, ret2nd;
	FT_HANDLE   usb_handle ;
	int     result;
	int     erron;
	int     old_vid, old_pid;
	char    *serialnum ;
	int     err_cnt;
	FT_STATUS   ret, ft_result;
	static const int chn_list[] = {
		USBMCA_CMDTYPE_CH1, USBMCA_CMDTYPE_CH2,
		USBMCA_CMDTYPE_CH3, USBMCA_CMDTYPE_CH4
	};
	USBMCA_STATUS   status;
	unsigned qword  meas_msec, meas_count;

	erron = 0 ;
	usb_handle = NULL ;

	serialnum = USBMCA_SERIALNUM ;

	memset(&histdt, 0, sizeof(histdt));

	/********** setting id(vendor, product) **********/
	old_vid = 0 ;
	old_pid = 0 ;
	ret1st = usbmca_get_vidpid(&old_vid, &old_pid) ;
	ret2nd = usbmca_set_vidpid(USBMCA_ID_VENDOR, USBMCA_ID_PRODUCT) ;

	if((ret1st | ret2nd) != FT_OK) {
		printf("setting ID(vendor + product) error(%u)...\n", ret1st) ;
		erron = -11 ;
	}

	/*** device connect(by serial number) ***/
	if(erron == 0) {
		ret0th = usbmca_open_by_serialnum(serialnum, &usb_handle);

		if(ret0th != FT_OK) {
			printf("handle open error(%s, %u)...\n", serialnum, ret0th) ;
			erron = -31 ;
		}
	}

	/*** device initial ***/
	if(erron == 0) {
		ret0th = usbmca_initial(usb_handle) ;

		if(ret0th != FT_OK) {
			printf("device initial error(%s, %u)...\n", serialnum, ret0th) ;
			erron = -41 ;
		}

		msleep(500);   /* wait initial complete.. */
	}
	
	/*** setting config ***/
	err_cnt = 0;
    /*****config, all CH*****/
    for (int i = 0; i < (int)(_countof(chn_list)); i++) {
		/* ADC gain = 4096ch */
		ret = usbmca_order_command_lite(usb_handle, chn_list[i], USBMCA_CMD_ADG, USBMCA_ADG_04K);
		if (ret > 0) err_cnt++;

		/* threshold = 10 ch */
		ret = usbmca_order_command_lite(usb_handle, chn_list[i], USBMCA_CMD_THR, 10);
		if (ret > 0) err_cnt++;

		/* lower level discrimination = 10ch */
		ret = usbmca_order_command_lite(usb_handle, chn_list[i], USBMCA_CMD_LLD, 10);
		if (ret > 0) err_cnt++;

		/* upper level discrimination = 4090ch */
		ret = usbmca_order_command_lite(usb_handle, chn_list[i], USBMCA_CMD_ULD, 4090);
		if (ret > 0) err_cnt++;

		/* offset level = 0ch */
		ret = usbmca_order_command_lite(usb_handle, chn_list[i], USBMCA_CMD_OFS, 0);
		if (ret > 0) err_cnt++;
    }
	
    /* PDS: fast mode */
    ret = usbmca_order_command_lite(usb_handle, USBMCA_CMDTYPE_CMN, USBMCA_CMD_PDSW, USBMCA_PDS_FAST);
	if (ret > 0) err_cnt++;
	
	if (err_cnt > 0) {
		printf("device setting config error(%s, %d)...\n", serialnum, err_cnt) ;
		erron = -51 ;
	}

	/*** run measurement ***/
	result = 0 ;
	if(erron == 0) {
		err_cnt = 0;

		/* save measurment info */
		memset(&histdt, 0, sizeof(histdt));
		histdt.meas_time = MEASURING_TIME;
		
		for (int i = 0; i < (int)(_countof(histdt.adg)); i++) {
			histdt.adg[i] = usbmca_get_adg_maxch(USBMCA_ADG_04K);
		}
		ft_result = usbmca_stop_measure(usb_handle);    /* measuring stop */
		if (ft_result > 0) err_cnt++;
		
		meas_count = usbmca_msec2count(MEASURING_TIME); /* convert digit time */
		ft_result = usbmca_set_measuretime(usb_handle, meas_count);  /* setting measure time */
		if (ft_result > 0) err_cnt++;
		
		ft_result = usbmca_set_measmode(usb_handle, USBMCA_MODE_HIST);       /* histogram mode */
		if (ft_result > 0) err_cnt++;

		ft_result = usbmca_set_timemode(usb_handle, USBMCA_MODE_REALTIME);   /* real time */
		if (ft_result > 0) err_cnt++;

		ft_result = usbmca_clear_data(usb_handle);      /* measurement data clear */
		if (ft_result > 0) err_cnt++;

		ft_result = usbmca_start_measure(usb_handle);   /* measuring start */
		if (ft_result > 0) err_cnt++;
		
		while (1) {
			/* read status */
			memset(&status, 0, sizeof(status));
			ret0th = usbmca_read_status(usb_handle, &status);
			
			if (ret0th < 0) {
			    /* device read error(status)... */
			    result = -1;
			    break;
			}
			
			/* read histogram data */
			for (int i = 0; i < (int)(_countof(histdt.data)); i++) {
			    ret0th = usbmca_read_histogram(usb_handle, (i + 1), histdt.adg[i], histdt.data[i]);

			    if (ret0th < 0) {
			        /* device read error(histogram data)... */
			        result = -2;
			        break;
			    }

			    /* total count */
			    histdt.totoal[i] = 0llu;

			    for (int j = 0; j < (int)(_countof(histdt.data[i])); j++) {
			        histdt.totoal[i] += (unsigned qword)(histdt.data[i][j]);
			    }
			}

			meas_msec = usbmca_count2msec(status.real_time);    /* convert msec */
			printf("elapsed time: %6llu msec\n", meas_msec);
			printf("    CH1: throuput rate = %6d, count = %6d.\n"
			       "    CH2: throuput rate = %6d, count = %6d.\n"
			       "    CH3: throuput rate = %6d, count = %6d.\n"
			       "    CH4: throuput rate = %6d, count = %6d.\n\n",
			    status.ch[0].rate.throughput, status.ch[0].count.throughput,
			    status.ch[1].rate.throughput, status.ch[1].count.throughput,
			    status.ch[2].rate.throughput, status.ch[2].count.throughput,
			    status.ch[3].rate.throughput, status.ch[3].count.throughput);

			/* check measurement stop */
			if (meas_msec >= MEASURING_TIME) {
			    file_save_histogram(SAVE_FILE, &histdt);
			    break;
			}

			msleep(250);    /* waiting */
		}

		usbmca_stop_measure(usb_handle) ;       /* stop */

		if (result == 0) {
			printf("complete!!!\n");
		}
		else {
			printf("measuring error(%s, %d)...\n", serialnum, result);
			erron = -61;
        }
    }

	/*** device disconnect ***/
	if(usb_handle != 0) {
		usbmca_close(usb_handle) ;
	}
   
	/********** restore id(vendor, product) **********/
	usbmca_set_vidpid(old_vid, old_pid) ;

	return  (EXIT_SUCCESS) ;
}

/* #########################################################################
    Name      : file_save_histogram
    Function  : write data to the text file
    Arguments :
        in savefile : file name
        in info     : save data
    Return :
         0 : success
        -1 : failed to open file
   ######################################################################### */
int file_save_histogram(char *savefile, USBMCA_HISTOGRAM *info)
{
    FILE    *fp;
    int     i, j;
    int     result;
    char    text[1024], text0[256];

    result = 0;

    memset(text, 0, sizeof(text));
    memset(text0, 0, sizeof(text0));

    fp = fopen(savefile, "wt");

    if (fp != NULL) {
        /* header save */
        fprintf(fp, "[header]\n");
        fprintf(fp, "meas time,%llu  #nsec\n", info->meas_time);

        /* total count */
        fprintf(fp, "total count");

        for (i = 0; i < (int)(_countof(info->totoal)); i++) {
            fprintf(fp, ",%lld", info->totoal[i]);
        }

        fprintf(fp, "\n");

        /* adg gain */
        fprintf(fp, "ADG gain");

        for (i = 0; i < (int)(_countof(info->adg)); i++) {
            fprintf(fp, ",%d", info->adg[i]);
        }

        fprintf(fp, "\n\n");

        /* data(histogram) */
        fprintf(fp, "[data]\n");
        fprintf(fp, "#ch,CH1,CH2,CH3,CH4\n");

        /* data(histogram) save */
        for (i = 0; i < (int)(_countof(info->data[0])); i++) {
            sprintf(text, "%d", i);

            for (j = 0; j < (int)(_countof(info->data)); j++) {
                sprintf(text0, ",%d", info->data[j][i]);
                strcat(text, text0);
            }

            fprintf(fp, "%s\n", text);
        }

        fclose(fp);
    }
    else {
        result = -1;
    }


    return  result;

}

