//*****************************************************************************
// Description     : Main sample program of list mode
// Target Device   : APG7400A
// Target Platform : Linux, C
// Date            : June 2025
// Version         : 2.0
//*****************************************************************************
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "ftd2xx.h"
#include "usbmca.h"

//
// Modify the following defines according to your environment
//
/******************************************************************************/
#define     MEASURING_TIME      (3000llu)   /* msec */
#define     SAVE_FILE           "./list_data.bin"  // name of the file where list data saved
/******************************************************************************/
static int read_listdata(FT_HANDLE usb_handle, FILE *fp);

//#########################################################################
//  Name      : main
//  Function  : entry point
//  Arguments :
//      in argc : number of parameters
//      in argv : parameters
//  Return :
//      0 : success
//      non-0 : failure
//  Note : 
//#########################################################################
int main(int argc, char **argv)
{
	FT_STATUS   ret0th, ret1st, ret2nd;
	FT_HANDLE   usb_handle ;
	int     result ;
	int     erron;
	int     old_vid, old_pid ;
	char    *serialnum ;
	unsigned qword  read_bytes;
	int     err_cnt;
	FT_STATUS   ret, ft_result;
	static const int chn_list[] = {
		USBMCA_CMDTYPE_CH1, USBMCA_CMDTYPE_CH2,
		USBMCA_CMDTYPE_CH3, USBMCA_CMDTYPE_CH4
	};
	USBMCA_STATUS   status;
	unsigned qword  meas_msec, meas_count;
    FILE    *fp;

	erron = 0 ;
	usb_handle = NULL ;

	serialnum = USBMCA_SERIALNUM ;

	/********** setting id(vendor, product) **********/
	old_vid = 0 ;
	old_pid = 0 ;
	ret1st = usbmca_get_vidpid(&old_vid, &old_pid) ;
	ret2nd = usbmca_set_vidpid(USBMCA_ID_VENDOR, USBMCA_ID_PRODUCT) ;

	if((ret1st | ret2nd) != FT_OK) {
		printf("setting ID(vendor + product) error(%u)...\n", ret1st) ;
		erron = -11 ;
	}

	/*** device connect(by serial number) ***/
	if(erron == 0) {
		ret0th = usbmca_open_by_serialnum(serialnum, &usb_handle);

		if(ret0th != FT_OK) {
			printf("handle open error(%s, %u)...\n", serialnum, ret0th) ;
			erron = -31 ;
		}
	}

	/*** device initial ***/
	if(erron == 0) {
		ret0th = usbmca_initial(usb_handle) ;

		if(ret0th != FT_OK) {
			printf("device initial error(%s, %u)...\n", serialnum, ret0th) ;
			erron = -41 ;
		}

		msleep(500);   /* wait initial complete.. */
	}

	/*** setting config ***/
	err_cnt = 0;
	/*****config, all CH*****/
	for (int i = 0; i < (int)(_countof(chn_list)); i++) {
		/* ADC gain = 4096ch */
		ret = usbmca_order_command_lite(usb_handle, chn_list[i], USBMCA_CMD_ADG, USBMCA_ADG_04K);
		if (ret > 0) err_cnt++;

		/* threshold = 10 ch */
		ret = usbmca_order_command_lite(usb_handle, chn_list[i], USBMCA_CMD_THR, 10);
		if (ret > 0) err_cnt++;

		/* lower level discrimination = 10ch */
		ret = usbmca_order_command_lite(usb_handle, chn_list[i], USBMCA_CMD_LLD, 10);
		if (ret > 0) err_cnt++;

		/* upper level discrimination = 4090ch */
		ret = usbmca_order_command_lite(usb_handle, chn_list[i], USBMCA_CMD_ULD, 4090);
		if (ret > 0) err_cnt++;

		/* offset level = 0ch */
		ret = usbmca_order_command_lite(usb_handle, chn_list[i], USBMCA_CMD_OFS, 0);
		if (ret > 0) err_cnt++;

	}

	/* PDS: fast mode */
	ret = usbmca_order_command_lite(usb_handle, USBMCA_CMDTYPE_CMN, USBMCA_CMD_PDSW, USBMCA_PDS_FAST);
	if (ret > 0) err_cnt++;

	if (err_cnt > 0) {
		printf("device setting config error(%s, %d)...\n", serialnum, err_cnt) ;
		erron = -51 ;
	}
	
	
	// ===========================================================================
	// open file in advance
	fp = fopen(SAVE_FILE, "wb") ;
	if (fp == NULL) {
		printf("Failed to create file(%s)\n", SAVE_FILE);
		erron = -71;
	}	

	/*** run measurement ***/
	result = 0 ;
	if(erron == 0) {
		err_cnt = 0;
		read_bytes = 0llu;

		ft_result = usbmca_stop_measure(usb_handle);   /* measuring stop */
		if (ft_result > 0) err_cnt++;

		meas_count = usbmca_msec2count(MEASURING_TIME); /* convert digit time */
		ft_result = usbmca_set_measuretime(usb_handle, meas_count);  /* setting measure time */
		if (ft_result > 0) err_cnt++;

		ft_result = usbmca_set_measmode(usb_handle, USBMCA_MODE_LIST);       /* histogram mode */
		if (ft_result > 0) err_cnt++;

		ft_result = usbmca_set_timemode(usb_handle, USBMCA_MODE_REALTIME);   /* real time */
		if (ft_result > 0) err_cnt++;

		ft_result = usbmca_clear_data(usb_handle) ;     /* measurement data clear */
		if (ft_result > 0) err_cnt++;
		ft_result = usbmca_start_measure(usb_handle) ;  /* mesuring start */
		if (ft_result > 0) err_cnt++;
		
		
		while (1) {
			/* read status */
			memset(&status, 0, sizeof(status));
			ret0th = usbmca_read_status(usb_handle, &status);

			if (ret0th < 0) {
				/* device read error(status)... */
				result = -3;
				break;
			}
			
			ret0th = read_listdata(usb_handle, fp);
			
			if (ret0th < 0) {
				/* device read error(status)... */
				result = -2;
				break;
			}

			read_bytes += (unsigned qword)(ret0th);

			meas_msec = usbmca_count2msec(status.real_time);    /* convert msec */
			printf("elapsed time: %6llu msec\n", meas_msec);
			printf("    CH1: throuput rate = %6d, count = %6d.\n"
				"    CH2: throuput rate = %6d, count = %6d.\n"
				"    CH3: throuput rate = %6d, count = %6d.\n"
				"    CH4: throuput rate = %6d, count = %6d.\n\n",
			status.ch[0].rate.throughput, status.ch[0].count.throughput,
			status.ch[1].rate.throughput, status.ch[1].count.throughput,
			status.ch[2].rate.throughput, status.ch[2].count.throughput,
			status.ch[3].rate.throughput, status.ch[3].count.throughput);

			/* check measurement stop */
			if (meas_msec >= MEASURING_TIME) {
				ret0th = read_listdata(usb_handle, fp);

				if (ret0th > 0) {
					read_bytes += (unsigned qword)(ret0th);
				}

				break;
			}

			msleep(100);    /* waiting */
		}

		usbmca_stop_measure(usb_handle) ;       /* stop */

		if (read_bytes > 0) {
			printf("complete measurement(%llu byte(s))!!!\n", read_bytes);
		}
		else {
			printf("mesurement error(%s, %d)...\n", serialnum, result);
			erron = -61;
		}
        fclose(fp) ;
	}

	/*** device disconnect ***/
	if(usb_handle != 0) {
		usbmca_close(usb_handle) ;
	}

	/********** restore id(vendor, product) **********/
	usbmca_set_vidpid(old_vid, old_pid) ;

	return  (EXIT_SUCCESS) ;
}

/* #########################################################################
    Name      : read_listdata
    Function  : read list data
    Arguments :
        in usb_handle : USB handle
        in save_file  : save file
    Return :
        0~ : success
        -1 : failed to request list data
        -2 : failed to save list data
        -3 : invalid read size of list data
        -4 : failed to read list data
   ######################################################################### */
static int read_listdata(FT_HANDLE usb_handle, FILE *fp)
{
    FT_STATUS   ft0th, ft1st;
    int     result, ret0th;
    int     nbytes, nblock;
    int     rxsize, rxbytes, zbyte;
    char    rxdata[2048];

    size_t  abyte ;

    result = 0;

    memset(rxdata, 0, sizeof(rxdata));

    /* read list data */
    nbytes = 0;     /* read bytes   */
    nblock = 0;     /* read blocks  */
    ft0th = usbmca_read_listinfo(usb_handle, &nbytes, &nblock);

    /* data on */
    if (ft0th == FT_OK && nbytes > 0) {
        while (1) {
            zbyte = nbytes - result;

            if (zbyte >= (int)(sizeof(rxdata))) {
                rxsize = sizeof(rxdata);
            }
            else {
                rxsize = zbyte;
            }

            rxbytes = 0;
            ft1st = usbmca_read_data(usb_handle, rxdata, rxsize, &rxbytes);

            if (ft1st != FT_OK) {
                result = -4;
                break;
            }
            else if (rxsize != rxbytes) {
                result = -3;
                break;
            }
            else {
                result += rxbytes;

                ret0th = 0 ;

                /* write  data */
                abyte  =  fwrite(rxdata, (size_t)(1), (size_t)(rxbytes), fp) ;
                ret0th = (int)(abyte) ;

                if (ret0th < 0) {
                    result = -2;
                    break;
                }

                if (result >= nbytes) {
                    break;
                }
            }
        }
    }
    else {
        if (ft0th != FT_OK) {
            result = -1;    /* error on */
        }
    }

    return result;

}
