//*****************************************************************************
// Description     : USB-MCA(APG7400A) control module
// Target Device   : APG7400A
// Target Platform : Linux, C
// Date            : July 2017
// Version         : 1.0
//*****************************************************************************
#include    <stdio.h>
#include    <stdlib.h>
#include    <string.h>
#include    <sys/socket.h>
#include    <netinet/in.h>

#include    "ftd2xx.h"

#include    "usbmca.h"

/* #########################################################################
    Name      : usbmca_get_vidpid
    Function  : get the product ID
    Arguments :
        in id_vendor  : vendor ID
        in id_product : product ID
    Return :
        Please refer to the software application development
               D2XX programmer's guide
   ######################################################################### */
FT_STATUS usbmca_get_vidpid(int *id_vendor, int *id_product)
{
    FT_STATUS   result ;
    DWORD   wvendor, wproduct ;
    
    wvendor  = (DWORD)(0) ;
    wproduct = (DWORD)(0) ;
    result   = FT_GetVIDPID(&wvendor, &wproduct) ;
    
    *id_vendor  = (int)(wvendor) ;      /* vender ID  */
    *id_product = (int)(wproduct) ;     /* product ID */
    
    return  result ;
    
}


/* #########################################################################
    Name      : usbmca_set_vidpid
    Function  : switch to the product to be used (set product ID)
    Arguments :
        in id_vendor  : vendor ID
        in id_product : product ID
    Return :
        Please refer to the software application development
               D2XX programmer's guide
   ######################################################################### */
FT_STATUS usbmca_set_vidpid(int id_vendor, int id_product)
{
    /* setting vendor ID, product ID */
    return  FT_SetVIDPID((DWORD)(id_vendor), (DWORD)(id_product)) ;
}


/* #########################################################################
    Name      : usbmca_create_deviceinfo
    Function  : create a list of connected devices
    Arguments :
        in count  : number of devices
        in result : it shows the result of creating the device list
                     0 = success
                    -1 = failed to acquire the number of connected devices
                    -2 = failure to secure dynamic area
                    -3 = device list generation failed
    Return :
            address where the device list is stored
   ######################################################################### */
FT_DEVICE_LIST_INFO_NODE* usbmca_create_deviceinfo(int *count, int *result)
{
    FT_DEVICE_LIST_INFO_NODE    *pinfo ;
    FT_STATUS   ret1st, ret2nd ;
    int     nbyte ;
    DWORD   wcount ;
    
    pinfo = NULL ;
    
    *count  = 0 ;
    *result = 0 ;
    
    /* create device list */
    wcount = 0 ;
    ret1st = FT_CreateDeviceInfoList(&wcount);
    
    if(ret1st == FT_OK && wcount > 0) {
        /* create alloc area */
        nbyte = (int)((wcount) * sizeof(FT_DEVICE_LIST_INFO_NODE)) ;
        pinfo = malloc((size_t)(nbyte)) ;
        
        if(pinfo != NULL) {
            /* get device list */
            memset(pinfo, 0, nbyte) ;
            ret2nd = FT_GetDeviceInfoList(pinfo, &wcount) ;
            
            if(ret2nd != FT_OK) {
                *result = -3 ;
            }
        }
        else {
            *result = -2 ;
        }
    }
    else {
        if(ret1st != FT_OK) {
            *result = -1 ;
        }
    }
    
    if(*result < 0) {
        /* error -> free alloc area */
        usbmca_free_deviceinfo(pinfo) ;
        pinfo = NULL ;
    }
    
    *count = (int)(wcount) ;
    
    return  pinfo ;
    
}


/* #########################################################################
    Name      : usbmca_free_deviceinfo
    Function  : free device list area
    Arguments :
        in pinfo : the area where the device list is stored
    Return :
        Please refer to the software application development
               D2XX programmer's guide
   ######################################################################### */
FT_STATUS usbmca_free_deviceinfo(FT_DEVICE_LIST_INFO_NODE *pinfo)
{
    FT_STATUS   result ;
    
    result = FT_OK ;
    
    if(pinfo != NULL) {
        free((void*)(pinfo)) ;
    }
    else {
        result = FT_OTHER_ERROR ;
    }
    
    return  result ;
    
}


/* #########################################################################
    Name      : usbmca_open
    Function  : open usb-mca device
    Arguments :
        in mode   : means of opening devices
                    USBMCA_OPEN_INDEX     = open by index specification
                    USBMCA_OPEN_SERIALNUM = open by serial number
        in wparam : pass parameters according to the means(value of int)
                    USBMCA_OPEN_INDEX     = index number(pointer of char)
                    USBMCA_OPEN_SERIALNUM = serial number
        out usb_handle : identifier indicating USB-MCA device
    Return :
        Please refer to the software application development
               D2XX programmer's guide
   ######################################################################### */
FT_STATUS usbmca_open(int mode, void *param, FT_HANDLE *usb_handle)
{
    FT_STATUS   result ;
	unsigned qword index;
    
    result = FT_OK ;
    
    *usb_handle = 0 ;
    
    switch(mode) {
        case USBMCA_OPEN_INDEX:         /* device open by index */
			index = (unsigned qword)(param);
			result = FT_Open((int)(index), usb_handle);
            break ;
        case USBMCA_OPEN_SERIALNUM:     /* device open by serial number */
            result = FT_OpenEx((PVOID)(param), FT_OPEN_BY_SERIAL_NUMBER, usb_handle) ;
            break ;
        default:                        /* mode error... */
            result = FT_OTHER_ERROR ;
            break ;
    }
    
    return  result ;
    
}


/* #########################################################################
    Name      : usbmca_close
    Function  : close usb-mca device
    Arguments :
        in usb_handle : identifier indicating USB-MCA device
    Return :
        Please refer to the software application development
               D2XX programmer's guide
   ######################################################################### */
FT_STATUS usbmca_close(FT_HANDLE usb_handle)
{
    return  FT_Close(usb_handle) ;
}


/* #########################################################################
    Name      : usbmca_initial
    Function  : initialize the USB-MCA device
    Arguments :
        in usb_handle : identifier indicating USB-MCA device
    Return :
        Please refer to the software application development
               D2XX programmer's guide
   ######################################################################### */
FT_STATUS usbmca_initial(FT_HANDLE usb_handle)
{
    FT_STATUS   result ;
    
    result = FT_ResetDevice(usb_handle) ;   /* reset device */
    
    if(result == FT_OK) { result = FT_SetBitMode(usb_handle, 0xff, FT_BITMODE_RESET) ; }    /* setting bit mode */
    if(result == FT_OK) { result = FT_SetLatencyTimer(usb_handle, 2) ; }                    /* latency timer */
    if(result == FT_OK) { result = FT_SetTimeouts(usb_handle, 100, 100) ; }                 /* setting time out */
    if(result == FT_OK) { result = FT_Purge(usb_handle, (FT_PURGE_RX | FT_PURGE_TX)) ; }    /* data clear */
    
    return  result ;
    
}


/* #########################################################################
    Name      : usbmca_purge
    Function  : purge TX / RX data
    Arguments :
        in usb_handle : identifier indicating USB-MCA device
    Return :
        Please refer to the software application development
               D2XX programmer's guide
   ######################################################################### */
FT_STATUS usbmca_purge(FT_HANDLE usb_handle)
{
    /* purge usb buffer */
    return  FT_Purge(usb_handle, (FT_PURGE_RX | FT_PURGE_TX)) ;
}


/* #########################################################################
    Name      : usbmca_create_command
    Function  : create command message with measurement parameter
    Arguments :
        out format  : The created command message
        in  chn     : CH number
        in  command : command string
        in  value   : parameter value
    Return :
         0 : success
        -1 : CH number is invalid
        -2 : invalid command string length
   ######################################################################### */
int usbmca_create_command(USBMCA_FORMAT_COMMAND *format, int chn, char *command, unsigned int value)
{
    int     result ;
    int     length ;
    char    wcmnd[256] ;
    
    result = 0 ;
    
    memset(wcmnd,  0, sizeof(wcmnd)) ;
    memset(format, 0, sizeof(USBMCA_FORMAT_COMMAND)) ;
    
    switch(chn) {
        case USBMCA_CMDTYPE_CMN:    /* common */
            strcpy(wcmnd, command) ;
            break ;
        case USBMCA_CMDTYPE_CH1:    /* CH1 */
            sprintf(wcmnd, "%sW", command) ;

            if (strcmp(command, USBMCA_CMD_IOF) == 0 ||
                strcmp(command, USBMCA_CMD_IFG) == 0) {
                sprintf(wcmnd, "%d", (chn - 1));
            }

            break ;
        case USBMCA_CMDTYPE_CH2:    /* CH2 */
        case USBMCA_CMDTYPE_CH3:    /* CH3 */
        case USBMCA_CMDTYPE_CH4:    /* CH4 */
            sprintf(wcmnd, "%s%d", command, (chn - 1)) ;
            break ;
        default:                    /* failure */
            result = -1 ;
            break ;
    }
    
    if(result == 0) {
        length = strlen(wcmnd) ;
        
        if(length <= 4) {
            memcpy(format->command, wcmnd, 4) ;
            format->value = htonl(value) ;
        }
        else {
            result = -2 ;
        }
    }
    
    return  result ;
    
}


/* #########################################################################
    Name      : usbmca_write_command
    Function  : write command to usb-mca
    Arguments :
        in usb_handle : identifier indicating USB-MCA device
        in chn        : CH number
        in command    : command string
        in value      : parameter value
    Return :
        Please refer to the software application development
               D2XX programmer's guide
   ######################################################################### */
FT_STATUS usbmca_write_command(FT_HANDLE usb_handle, int chn, char *command, unsigned int value)
{
    USBMCA_FORMAT_COMMAND   tx ;
    FT_STATUS   result ;
    DWORD       txbyte ;
    int         ret ;
    
    result = FT_OK ;
    
    /* create command text */
    memset(&tx, 0, sizeof(tx)) ;
    ret = usbmca_create_command(&tx, chn, command, value) ;
    
    if(ret == 0) {
        /* command write */
        txbyte = 0 ;
        result = FT_Write(usb_handle, &tx, sizeof(tx), &txbyte) ;
        
        if(result == FT_OK) {
            if(txbyte != sizeof(tx)) {
                result = FT_OTHER_ERROR ;
            }
        }
    }
    else {
        result = FT_OTHER_ERROR ;
    }
    
    return  result ;
    
}


/* #########################################################################
    Name      : usbmca_read_command
    Function  : read response to command
    Arguments :
        in usb_handle : identifier indicating USB-MCA device
        in command    : response command
    Return :
        Please refer to the software application development
               D2XX programmer's guide
   ######################################################################### */
FT_STATUS usbmca_read_command(FT_HANDLE usb_handle, USBMCA_FORMAT_COMMAND *command)
{
    USBMCA_FORMAT_COMMAND   rx ;
    FT_STATUS   result ;
    DWORD       rxbyte ;
    
    result = FT_OK ;
    
    memset(&rx, 0, sizeof(rx)) ;
    
    rxbyte = 0 ;
    result = FT_Read(usb_handle, &rx, sizeof(rx), &rxbyte) ;
    
    if(result == FT_OK) {
        if(rxbyte == sizeof(rx)) {
            /* to host byte order */
            rx.value = htonl(rx.value) ;
        }
        else {
            result = FT_OTHER_ERROR ;
        }
    }
    
    command[0] = rx ;
    
    return  result ;
    
}


/* #########################################################################
    Name      : usbmca_order_command
    Function  : commands the request to USB-MCA
    Arguments :
        in  info  : request command information
        in  value : value of response command
        out ack   : response command
    Return :
        Please refer to the software application development
               D2XX programmer's guide
   ######################################################################### */
FT_STATUS usbmca_order_command(USBMCA_PARAMS *info, unsigned int *value, USBMCA_FORMAT_COMMAND *ack)
{
    USBMCA_FORMAT_COMMAND   rx ;
    FT_STATUS   result ;
    FT_STATUS   ret1st, ret2nd ;
    
    result = FT_OK ;
    
    value[0] = 0;
    memset(&rx, 0, sizeof(rx)) ;
    
    /* write command */
    ret1st = usbmca_write_command(info->usb_handle, info->channel,
                                  info->command,    info->value) ;
    if(ret1st == FT_OK) {
        /* read answer */
        ret2nd = usbmca_read_command(info->usb_handle, &rx) ;
        
        if(ret2nd == FT_OK) {
            *value = rx.value ;
        }
        else {
            result = ret2nd ;
        }
    }
    else {
        result = ret1st ;
    }
    
    if(ack != NULL) {
        ack[0] = rx ;
    }
    
    return  result ;
    
}


/* #########################################################################
    Name      : usbmca_order_command_lite
    Function  : commands the request to USB-MCA(lite version)
    Arguments :
        in usb_handle : identifier indicating USB-MCA device
        in chn        : CH number
        in command    : command string
        in value      : parameter value
    Return :
        Please refer to the software application development
               D2XX programmer's guide
   ######################################################################### */
FT_STATUS usbmca_order_command_lite(FT_HANDLE usb_handle, int channel, char *command, unsigned int value)
{
    USBMCA_FORMAT_COMMAND   ack ;
    USBMCA_PARAMS   params ;
    FT_STATUS       result ;
    unsigned int    ack_value;
    
    ack_value = 0;
    memset(&ack, 0, sizeof(ack)) ;
    
    memset(&params, 0, sizeof(params)) ;
    params.usb_handle = usb_handle ;    /* device handle */
    params.command    = command ;       /* command text */
    params.channel    = channel ;       /* CH number */
    params.value      = value;          /* value */
    result = usbmca_order_command(&params, &ack_value, &ack) ;
    
    return  result ;
    
}


/* #########################################################################
    Name      : usbmca_read_data
    Function  : read measurement data from USB-MCA
    Arguments :
        in  usb_handle : identifier indicating USB-MCA device
        out rxbuffer   : address of buffer storing data
        in  rxsize     : capacity of buffer (byte)
        out rxbyte     : size of actually read data(byte)
    Return :
        Please refer to the software application development
               D2XX programmer's guide
   ######################################################################### */
FT_STATUS usbmca_read_data(FT_HANDLE usb_handle, void *rxbuffer, int rxsize, int *rxbyte)
{
    FT_STATUS   result;
    DWORD       read_byte ;
    char        *p ;
    int         rxcount ;
    int         rxbytes ;
    int         zbyte ;
    
    result = FT_OK ;
    
    rxcount = 0 ;
    rxbytes = 0 ;
    p = (char*)(rxbuffer) ;
    
    zbyte = rxsize ;
    
    while(1) {
        if (rxbytes >= rxsize) {
            break;
        }

        read_byte = 0 ;
        result = FT_Read(usb_handle, &p[rxbytes], zbyte, &read_byte) ;
        
        if(result != FT_OK) {
            break ;
        }
        
        rxbytes += (int)(read_byte) ;
        zbyte = (DWORD)(rxsize - rxbytes) ;
        rxcount++ ;
    }
    
    *rxbyte = rxbytes ;
    
    return  result ;
    
}


/* #########################################################################
    Name      : usbmca_read_listinfo
    Function  : request list data
    Arguments :
        in  usb_handle : identifier indicating USB-MCA device
        in  nbyte      : capacity of list data at present (byte)
        in  nblock     : number of list data at the present time
    Return :
        Please refer to the software application development
               D2XX programmer's guide
   ######################################################################### */
FT_STATUS usbmca_read_listinfo(FT_HANDLE usb_handle, int *nbyte, int *nblock)
{
    USBMCA_FORMAT_COMMAND   rx ;
    USBMCA_PARAMS   param ;
    FT_STATUS   result ;
    unsigned int    value ;
    
    result = FT_OK ;
    
    *nbyte  = 0 ;
    *nblock = 0 ;
    
    value = 0 ;
    memset(&rx, 0, sizeof(rx)) ;
    
    /* write request list data command */
    param.usb_handle = usb_handle ;
    param.channel    = USBMCA_CMDTYPE_CMN;
    param.command    = USBMCA_CMD_LISR ;
    result = usbmca_order_command(&param, &value, &rx) ;
    
    if(result == FT_OK) {
        /* check answer... */
        if(memcmp(rx.command, USBMCA_CMD_LISA, sizeof(rx.command)) == 0) {
            /* data size */
            *nblock = (int)(rx.value) ;     /* block unit   */
            *nbyte  = *nblock * 8 ;         /* byte size    */
        }
        else {
            result = FT_OTHER_ERROR ;
        }
    }
    
    return  result ;
    
}


/* #########################################################################
    Name      : usbmca_read_histogram
    Function  : read histogram data
    Arguments :
        in  usb_handle : identifier indicating USB-MCA device
        in  channels   : CH number
        in  maxch      : number of pins to read data (adjust to the ADC gain)
        out rxbuffer   : address of buffer storing data
    Return :
             0 = success
            -1 = switching of CH failed
         -1nnn = failed to request histogram data
         -2nnn = failed to read histogram data
         -3nnn = The size of the read data is invalid
   ######################################################################### */
int usbmca_read_histogram(FT_HANDLE usb_handle, int channel, int maxch, void *rxbuffer)
{
    USBMCA_FORMAT_COMMAND   rx ;
    USBMCA_PARAMS   param ;
    FT_STATUS   ret1st, ret2nd ;
    int     rxbyte ;
    int     i, j ;
    int     result ;
    int     rxbytes, rxcount ;
    char    *p ;
    char    command[32] ;
    unsigned int    value ;
    unsigned int    rxdata[USBMCA_MAX_RXHIST_CHN] ;
    
    result = 0 ;
    
    memset(&rx,    0, sizeof(rx)) ;
    memset(&param, 0, sizeof(param)) ;
    
    /* change CH */
    value  = 0 ;
    param.usb_handle = usb_handle ;
    param.command    = USBMCA_CMD_HCHW ;
    param.channel    = USBMCA_CMDTYPE_CMN;
    param.value = (unsigned int)(channel - 1) ;
    ret1st = usbmca_order_command(&param, &value, &rx) ;
    
    if(ret1st == FT_OK) {
        p = (char*)(rxbuffer) ;
        rxbytes = 0 ;
        memset(rxdata,  0, sizeof(rxdata)) ;
        memset(command, 0, sizeof(command)) ;
        
        /* calc read count */
        rxcount = (maxch * sizeof(int)) / (USBMCA_MAX_RXHIST_CHN * sizeof(int));

        for(i = 0 ; i < rxcount ; i++) {
            /* histogram request command */
            sprintf(command, USBMCA_CMD_HIXX, i) ;
            ret2nd = usbmca_write_command(usb_handle, USBMCA_CMDTYPE_CMN, command, 0) ;
            
            if(ret2nd != FT_OK) {
                /* error */
                result = (1001 + i) * (-1) ;
                break ;
            }
            
            /* read histogram data */
            rxbyte = 0 ;
            ret2nd = usbmca_read_data(usb_handle, rxdata, sizeof(rxdata), &rxbyte);
            
            if(ret2nd != FT_OK || rxbyte != sizeof(rxdata)) {
                if(ret2nd != FT_OK) {
                    /* error,  request command */
                    result = (2001 + i) * (-1) ;
                }
                else {
                    /* error, read size */
                    result = (3001 + i) * (-1) ;
                }
                
                break ;
            }
            
            for(j = 0 ; j < (int)(_countof(rxdata)) ; j++) {
                /* to host byte order */
                rxdata[j] = ntohl(rxdata[j]) ;
            }
            
            memcpy(&p[rxbytes], rxdata, sizeof(rxdata)) ;
            rxbytes += (int)(rxbyte) ;
        }
    }
    else {
        /* error, change CH */
        result = -1 ;
    }
    
    return  result ;
    
}


/* #########################################################################
    Name      : usbmca_read_status
    Function  : read the measurement status
    Arguments :
        in  usb_handle : identifier indicating USB-MCA device
        out status     : Address of the buffer storing the measurement status
    Return :
             0 = success
            -1 = measurement status request failed
            -2 = failed to read data
            -3 = The size of the read data is invalid
   ######################################################################### */
int usbmca_read_status(FT_HANDLE usb_handle, USBMCA_STATUS *status)
{
    USBMCA_STATUS_RAW   status_raw ;
    FT_STATUS   ret1st, ret2nd ;
    int     rxbyte ;
    int     result ;
    int     i ;
    
    result = 0 ;
    
    memset(status,      0, sizeof(status[0])) ;
    memset(&status_raw, 0, sizeof(status_raw)) ;
    
    ret1st = usbmca_write_command(usb_handle, USBMCA_CMDTYPE_CMN, USBMCA_CMD_STUW, 0) ;
    
    if(ret1st == FT_OK) {
        rxbyte = 0 ;
        ret2nd = usbmca_read_data(usb_handle, &status_raw, sizeof(status_raw), &rxbyte);
        
        if(ret2nd == FT_OK) {
            if(rxbyte == sizeof(status_raw)) {
                usbmca_memcpy(&status->real_time, 2, status_raw.real_time, 0, sizeof(status_raw.real_time)) ;
                status->real_time = usbmca_ntoh64(status->real_time) ;
                
                for(i = 0 ; i < (int)(_countof(status->ch)) ; i++) {
                    usbmca_memcpy(&status->ch[i].time.dead,  2, status_raw.ch[i].dead_time,  0,
                                   sizeof(status_raw.ch[i].dead_time)) ;
                    usbmca_memcpy(&status->ch[i].time.live,  2, status_raw.ch[i].live_time,  0,
                                   sizeof(status_raw.ch[i].live_time)) ;
                    usbmca_memcpy(&status->ch[i].rate.input, 1, status_raw.ch[i].rate_input, 0,
                                   sizeof(status_raw.ch[i].rate_input)) ;
                    usbmca_memcpy(&status->ch[i].rate.throughput,  1, status_raw.ch[i].rate_throughput,  0,
                                   sizeof(status_raw.ch[i].rate_throughput)) ;
                    usbmca_memcpy(&status->ch[i].count.throughput, 0, status_raw.ch[i].count_throughput, 0,
                                   sizeof(status_raw.ch[i].count_throughput)) ;
                    
                    status->ch[i].time.dead        = usbmca_ntoh64(status->ch[i].time.dead) ;
                    status->ch[i].time.live        = usbmca_ntoh64(status->ch[i].time.live) ;
                    status->ch[i].rate.input       = ntohl(status->ch[i].rate.input) ;
                    status->ch[i].rate.throughput  = ntohl(status->ch[i].rate.throughput) ;
                    status->ch[i].count.throughput = ntohl(status->ch[i].count.throughput) ;
                }
            }
            else {
                result = -3 ;
            }
        }
        else {
            result = -2 ;
        }
    }
    else {
        result = -1 ;
    }
    
    return  result ;
    
}


/* #########################################################################
    Name      : usbmca_set_measuretime
    Function  : set measurement time for USB-MCA
    Arguments :
        in  usb_handle : identifier indicating USB-MCA device
        in  time_count : measurement time (specified by digit)
    Return :
             0 = success
            -1 = failure in upper digit setting
            -2 = failure in lower digit setting
   ######################################################################### */
int usbmca_set_measuretime(FT_HANDLE usb_handle, unsigned qword time_count)
{
    FT_STATUS   ret1st, ret2nd ;
    int     result ;
    unsigned int    msb32bit, lsb32bit ;
    
    result = 0 ;
    
    msb32bit = usbmca_msb32bit(time_count) ;
    lsb32bit = usbmca_lsb32bit(time_count) ;
    
    /* write measurement time, MSB */
    ret1st = usbmca_order_command_lite(usb_handle, USBMCA_CMDTYPE_CMN, USBMCA_CMD_MT0W, msb32bit) ;
    
    if(ret1st == FT_OK) {
        /* write measurement time, LSB */
        ret2nd = usbmca_order_command_lite(usb_handle, USBMCA_CMDTYPE_CMN, USBMCA_CMD_MT1W, lsb32bit) ;
        
        if(ret2nd != FT_OK) {
            result = -2 ;
        }
    }
    else {
        result = -1 ;
    }
    
    return  result ;
    
}


/* #########################################################################
    Name      : usbmca_get_adg_maxch
    Function  : get pin(ch) count by ADC gain
    Arguments :
        in adc_gain : code of ADC gain
    Return :
         returns the specified number of pin(ch)s
   ######################################################################### */
int usbmca_get_adg_maxch(int adc_gain)
{
    int     i;
    int     result;
    static const int    gain_list[][2] = {
        { USBMCA_ADG_16K, 16384 },  /* 16384ch */
        { USBMCA_ADG_08K,  8192 },  /*  8192ch */
        { USBMCA_ADG_04K,  4096 },  /*  4096ch */
        { USBMCA_ADG_02K,  2048 },  /*  2048ch */
        { USBMCA_ADG_01K,  1024 },  /*  1024ch */
        { USBMCA_ADG_512,   512 },  /*   512ch */
        { -1, 0 }                   /* end of define */
    };

    result = 0;

    for (i = 0; gain_list[i][0] != -1; i++) {
        if (gain_list[i][0] == adc_gain) {
            result = gain_list[i][1];
            break;
        }
    }

    return  result;

}


/* #########################################################################
    Name      : usbmca_memcpy
    Function  : copy of memory
    Arguments :
        in dest       : start address of copy destination
        in dest_index : relative position of copy destination
        in src        : start address of copy source
        in src_index  : relative position of copy source
        in nbyte      : size to copy (byte)
    Return :
             0 = success
            -1 = Invalid specification of copy destination address
            -2 = Invalid specification of copy source address
            -3 = incorrect size to copy
   ######################################################################### */
int usbmca_memcpy(void *dest, int dest_index, void *src, int src_index, int nbyte)
{
    int     result ;
    char    *p, *q ;
    
    result = 0 ;
    
    if(dest != NULL && dest_index >= 0) {
        if(src != NULL && src_index >= 0) {
            if(nbyte > 0) {
                p = (char*)(src) ;
                q = (char*)(dest) ;
                memcpy(&q[dest_index], &p[src_index], nbyte) ;
            }
            else {
                result = -3 ;
            }
        }
        else {
            result = -2 ;
        }
    }
    else {
        result = -1 ;
    }
    
    return  result ;
    
}


/* #########################################################################
    Name      : usbmca_adjust_listdata
    Function  : adjust list data
    Arguments :
        in data  : RAW list data
        in list  : adjusted list data
    Return :
         nothing...
   ######################################################################### */
void usbmca_adjust_listdata(void *data, USBMCA_LIST_ADJUST *list)
{
    unsigned char   *p ;
    union {
        unsigned char   a08bit[8] ;
        unsigned short  a16bit[4] ;
        unsigned qword  u64bit ;
    } work ;
    
    p = (unsigned char*)(data) ;
    
    work.u64bit = (unsigned qword)(0) ;
    
    /* absolution count */
    memcpy(&work.a08bit[2], &p[0], 6) ;
    work.u64bit = usbmca_ntoh64(work.u64bit) ;
    list->abs = work.u64bit & (unsigned qword)(0x0fffffffffff) ;
    
    /* CH and pha */
    work.a08bit[0] = p[6] ;
    work.a08bit[1] = p[7] ;
    work.a16bit[0] = ntohs(work.a16bit[0]) ;
    list->chn = ((work.a16bit[0] >> 14) & 0x03) + 1 ;   /* CH number */
    list->pha =   work.a16bit[0] & 0x3fff ;             /* PHA */
}


/* #########################################################################
    Name      : usbmca_hton64
    Function  : convert 64 bit value to network byte order
    Arguments :
        in value : value in host byte order
    Return :
         value in netowork byte order
   ######################################################################### */
unsigned qword usbmca_hton64(unsigned qword value)
{
    int     i ;
    unsigned char   *p, *q ;
    unsigned qword  result ;
    static const char   ncheck[2] = { 0x12, 0x34 } ;
    
    result = value ;
    
    if(*(short*)(ncheck) != (short)(0x1234)) {      /* little endian machine */
        p = (unsigned char*)(&value) ;
        q = (unsigned char*)(&result) ;
        
        for(i = 0 ; i < (int)(sizeof(value)) ; i++) {
            q[i] = p[(sizeof(value)-1)-i] ;
        }
    }
    
    return  result ;
    
}

