//*****************************************************************************
// Description     : Definition of constants and structures etc.
//                   to be used for the USB-MCA (APG 7400 A) control module
// Target Device   : APG7400A
// Target Platform : Linux, C
// Date            : July 2017
// Version         : 1.0
//*****************************************************************************
#ifndef     __USBMCA_H__
#define     __USBMCA_H__

#include    "ftd2xx.h"
#include    "sys_compatible.h"

/******************************************************************/
/* Product information of USB-MCA (APG 7400 A) */
#define     USBMCA_ID_VENDOR    (0x1ca6)    /* vendor  ID   */
#define     USBMCA_ID_PRODUCT   (0x0001)    /* product ID   */
#define     USBMCA_SERIALNUM    ("APG7400AA")    /* serial number */

/******************************************************************/
/***** command string by CH number *****/
#define     USBMCA_CMD_ADG      "ADG"   /* ADC gain */
#define     USBMCA_CMD_THR      "THR"   /* threshold(ch) */
#define     USBMCA_CMD_LLD      "LLD"   /* lower level discrimination(ch) */
#define     USBMCA_CMD_ULD      "ULD"   /* upper level discrimination(ch) */
#define     USBMCA_CMD_OFS      "OFS"   /* offset(ch) */
#define     USBMCA_CMD_IOF      "IOF"   /* unuse... */
#define     USBMCA_CMD_IFG      "IFG"   /* unuse... */

/***** common command string of USB-MCA *****/
#define     USBMCA_CMD_MODW     "MODW"      /* measure mode command */
#define     USBMCA_CMD_MMDW     "MMDW"      /* time mode command */
#define     USBMCA_CMD_MT0W     "MT0W"      /* measurement time command(msb) */
#define     USBMCA_CMD_MT1W     "MT1W"      /* measurement time command(lsb) */
#define     USBMCA_CMD_PDSW     "PDSW"      /* peak search mode command */
#define     USBMCA_CMD_AQSW     "AQSW"      /* start mesure command */
#define     USBMCA_CMD_AQEW     "AQEW"      /* stop  mesure command */
#define     USBMCA_CMD_CLRW     "CLRW"      /* data clear command */
#define     USBMCA_CMD_HCHW     "HCHW"      /* histogram CH change command */
#define     USBMCA_CMD_HIXX     "HI%02X"    /* histogram command */

#define     USBMCA_CMD_LISR     "LISR"      /* list data request command    */
#define     USBMCA_CMD_LISA     "LISA"      /* list data size command(ack)  */

#define     USBMCA_CMD_STUW     "STUW"      /* status request command */


/******************************************************************/
/* constant associated with size or number */
#define     USBMCA_MAX_CHNUM            (  4)       /* Max CH                 */
#define     USBMCA_MAX_RXHIST_CNT       ( 32)       /* read histogram count   */
#define     USBMCA_MAX_RXHIST_CHN       (512)       /* read histogram points(1 times)	*/
                                                    /* total histogram points(ch)		*/
#define     USBMCA_MAX_HISTGRAM_CH  (USBMCA_MAX_RXHIST_CNT * USBMCA_MAX_RXHIST_CHN)
                                                    /* total bytes histogram(16384ch) */
#define     USBMCA_MAX_HISTGRAM_BYTE         (USBMCA_MAX_HISTGRAM_CH * sizeof(int))

#define     USBMCA_MAX_LISTDATA_BYTE         (8)    /* list data length(1 count) */

/******************************************************************/
/* constant accompanying the parameter */
#define     USBMCA_MODE_REALTIME    (0)     /* real time */
#define     USBMCA_MODE_LIVETIME    (1)     /* live time */

#define     USBMCA_MODE_HIST        (0)     /* histogram */
#define     USBMCA_MODE_LIST        (1)     /* list */

#define     USBMCA_PDS_ABS          (0)     /* abs mode */
#define     USBMCA_PDS_FAST         (1)     /* fast mode */

#define     USBMCA_UNIT_TMCOUNT     (40)    /* nano sec */
#define     USBMCA_SEC_TMCOUNT      (1000000000llu / (unsigned qword)USBMCA_UNIT_TMCOUNT)   /* 1sec time count */

/******************************************************************/
/* connection method */
#define     USBMCA_OPEN_INDEX           (1)		/* connection by index of device list */
#define     USBMCA_OPEN_SERIALNUM       (2)		/* connection by serial number        */

/******************************************************************/
/* ADG(ADC gain) */
#define     USBMCA_ADG_16K  (0)     /* 16384ch */
#define     USBMCA_ADG_08K  (1)     /*  8192ch */
#define     USBMCA_ADG_04K  (2)     /*  4096ch */
#define     USBMCA_ADG_02K  (3)     /*  2048ch */
#define     USBMCA_ADG_01K  (4)     /*  1024ch */
#define     USBMCA_ADG_512  (5)     /*   512ch */

/* Resolution upper limit and lower limit (ch) */
#define     USBMCA_DISCRIM_MIN  (    0)     /*     0 ch */
#define     USBMCA_DISCRIM_MAX  (16383)     /* 16383 ch */


/******************************************************************/
/* parameter value of CH number */
#define     USBMCA_CMDTYPE_CMN  (0)     /* common   */
#define     USBMCA_CMDTYPE_CH1  (1)     /* CH1      */
#define     USBMCA_CMDTYPE_CH2  (2)     /* CH2      */
#define     USBMCA_CMDTYPE_CH3  (3)     /* CH3      */
#define     USBMCA_CMDTYPE_CH4  (4)     /* CH4      */

/******************************************************************/
/* structure for generating command telegram */
typedef struct usbmca_params {
    FT_HANDLE       usb_handle ;
    char           *command ;
    int             channel ;
    unsigned int    value ;
} USBMCA_PARAMS ;

/* command telegram structure */
typedef struct usbmca_command_format {
    char            command[4] ;    /* command string */
    unsigned int    value ;         /* value          */
} USBMCA_FORMAT_COMMAND ;

/* structure that stores the measurement state after organizing it by item */
typedef struct usbmca_status {
    unsigned qword          real_time ;     /* elpased time     */
    char                    resv01[8] ;     /* reserve          */
    struct {
        struct {                            /* time             */
            unsigned qword  dead ;          /*   dead time      */
            unsigned qword  live ;          /*   live time      */
        } time ;
        struct {                            /* count            */
            unsigned int    throughput ;    /*   thruoghput     */
        } count ;
        struct {                            /* count rate       */
            unsigned int    input ;         /*   input          */
            unsigned int    throughput ;    /*   throughput     */
        } rate ;
        char                resv01[4] ;     /* reserve          */
    } ch[USBMCA_MAX_CHNUM] ;
} USBMCA_STATUS ;

/* structure storing the measurement state obtained from USB-MCA (before arrangement) */
typedef struct usbmca_status_raw {
    char        real_time[6] ;          /* elapsed time */
    struct {                            /* CH unit      */
        char    live_time[6] ;          /*   live time */
        char    dead_time[6] ;          /*   dead time */
        char    rate_throughput[3] ;    /*   throughput rate */
        char    count_throughput[4] ;   /*   throughput count */
        char    rate_input[3] ;         /*   input rate       */
    } ch[USBMCA_MAX_CHNUM] ;
} USBMCA_STATUS_RAW ;

/* structure that stores histogram data */
typedef struct usbmca_histogram {
    unsigned qword  meas_time;
    char            resv[8];
    int             adg[USBMCA_MAX_CHNUM];
    unsigned qword  totoal[USBMCA_MAX_CHNUM];
    unsigned int    data[USBMCA_MAX_CHNUM][USBMCA_MAX_HISTGRAM_CH];
} USBMCA_HISTOGRAM;

/* structure that stores list data after adjusting */
typedef struct usbmca_list_adjust {
    unsigned qword  abs ;           /* absolution count: real time */
    unsigned short  chn ;           /* channel number */
    unsigned short  pha ;           /* pulse height analisis */
    char            resv01[4] ;     /* reserve */
} USBMCA_LIST_ADJUST ;

/* Structure to store list data before adjusting */
typedef struct usbmca_list_raw {
    char    data[USBMCA_MAX_LISTDATA_BYTE] ;    /* data */
} USBMCA_LIST_RAW ;

/******************************************************************/
/* 64bit to msb 32bit */
#define     usbmca_msb32bit(value64)    \
    (unsigned int)(((unsigned qword)(value64) >> 32) & (unsigned qword)(0x0ffffffff))
/* 64bit to lsb 32bit */
#define     usbmca_lsb32bit(value64)    \
    (unsigned int)(((unsigned qword)(value64) >>  0) & (unsigned qword)(0x0ffffffff))

/* 64bit to 48bit */
#define     usbmca_msb48bit(value64)    \
    (unsigned qword)(((unsigned qword)(value64) >> 16) & (unsigned qword)(0x0ffffffffffff))


/******************************************************************/
#define     usbmca_hton64   usbmca_ntoh64   /* to host byte order */

/* macro to connect to device with serial number as key */
#define     usbmca_open_by_index(index, handle)         \
            usbmca_open(USBMCA_OPEN_INDEX, (void*)(index), handle)
/* macro to connect to device using index of device list as key */
#define     usbmca_open_by_serialnum(serialnum, handle) \
            usbmca_open(USBMCA_OPEN_SERIALNUM, (char*)(serialnum), handle)

/* macro to instruct measurement start */
#define     usbmca_start_measure(handle)    \
            usbmca_order_command_lite(handle, USBMCA_CMDTYPE_CMN, USBMCA_CMD_AQSW, 1)
/* macro to command measurement stop */
#define     usbmca_stop_measure(handle)     \
            usbmca_order_command_lite(handle, USBMCA_CMDTYPE_CMN, USBMCA_CMD_AQEW, 1)
/* macro to command measurement data clear */
#define     usbmca_clear_data(handle)       \
            usbmca_order_command_lite(handle, USBMCA_CMDTYPE_CMN, USBMCA_CMD_CLRW, 0)

/* macro to instruct switching of measurement mode(histogram or list...) */
#define     usbmca_set_measmode(handle, mode)   \
            usbmca_order_command_lite(handle, USBMCA_CMDTYPE_CMN, USBMCA_CMD_MODW, mode)
/* macro to instruct switching of measurement time mode(real or live) */
#define     usbmca_set_timemode(handle, mode)   \
            usbmca_order_command_lite(handle, USBMCA_CMDTYPE_CMN, USBMCA_CMD_MMDW, mode)

/******************************************************************/
/* nacros that convert time in nanoseconds */
#define     usbmca_nsec2count(time) \
            ((unsigned qword)(time) / (unsigned qword)USBMCA_UNIT_TMCOUNT)
/* nacros that convert time in milliseconds */
#define     usbmca_msec2count(time) \
            usbmca_nsec2count((unsigned qword)(time) * 1000000llu)
/* nacros that convert time in seconds */
#define     usbmca_sec2count(time)  \
            usbmca_msec2count((unsigned qword)(time) * 1000llu)

/* macros that convert time values to nanoseconds */
#define     usbmca_count2nsec(count)    \
            ((unsigned qword)(count) * (unsigned qword)USBMCA_UNIT_TMCOUNT)
/* macros that convert time values to ,olliseconds */
#define     usbmca_count2msec(count)    \
            (usbmca_count2nsec(count) / 1000000llu)
/* macros that convert time values to seconds */
#define     usbmca_count2sec(count) \
            (usbmca_count2msec(count) / 1000llu)

/******************************************************************/
#define     usbmca_write    FT_Write
#define     usbmca_read     FT_Read

/******************************************************************/
extern FT_STATUS usbmca_get_vidpid(int *id_vendor, int *id_product) ;
extern FT_STATUS usbmca_set_vidpid(int id_vendor, int id_product) ;

extern FT_DEVICE_LIST_INFO_NODE* usbmca_create_deviceinfo(int *count, int *result) ;
extern FT_STATUS usbmca_free_deviceinfo(FT_DEVICE_LIST_INFO_NODE *pinfo) ;

extern FT_STATUS usbmca_open(int mode, void *param, FT_HANDLE *usb_handle) ;
extern FT_STATUS usbmca_close(FT_HANDLE usb_handle) ;
extern FT_STATUS usbmca_initial(FT_HANDLE usb_handle) ;
extern FT_STATUS usbmca_purge(FT_HANDLE usb_handle) ;
extern int usbmca_create_command(USBMCA_FORMAT_COMMAND *format, int chn, char *command, unsigned int value) ;
extern FT_STATUS usbmca_write_command(FT_HANDLE usb_handle, int chn, char *command, unsigned int value) ;
extern FT_STATUS usbmca_read_command(FT_HANDLE usb_handle, USBMCA_FORMAT_COMMAND *command) ;
extern FT_STATUS usbmca_order_command(USBMCA_PARAMS *info, unsigned int *value, USBMCA_FORMAT_COMMAND *ack) ;
extern FT_STATUS usbmca_order_command_lite(FT_HANDLE usb_handle, int channel, char *command, unsigned int value) ;

extern FT_STATUS usbmca_read_data(FT_HANDLE usb_handle, void *rxbuffer, int rxsize, int *rxbyte) ;
extern int usbmca_read_histogram(FT_HANDLE usb_handle, int channel, int maxch, void *rxbuffer) ;
extern FT_STATUS usbmca_read_listinfo(FT_HANDLE usb_handle, int *nbyte, int *nblock) ;
extern int usbmca_read_status(FT_HANDLE usb_handle, USBMCA_STATUS *status) ;
extern int usbmca_set_measuretime(FT_HANDLE usb_handle, unsigned qword time_count) ;

extern int usbmca_get_adg_maxch(int adc_gain);
extern void usbmca_adjust_listdata(void *data, USBMCA_LIST_ADJUST *list);
extern int usbmca_memcpy(void *dest, int dest_index, void *src, int src_index, int nbyte) ;
extern unsigned qword usbmca_hton64(unsigned qword value) ;


#endif

